/*
    Histogram filtering tool.
    Copyright (C) 2006-2018 Darrick Wong

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "hist.h"

double getNumber(const char *string) {
	//int sz = strlen(string);
	char *last;
	double result = strtod(string, &last);

	switch(*last) {
		case 'G':	result *= 1000; /* fall through */
		case 'M':	result *= 1000; /* fall through */
		case 'K':	result *= 100; /* fall through */
		case 'D':	result *= 10; break;

		case 'n':	result /= 1000; /* fall through */
		case 'u':	result /= 1000; /* fall through */
		case 'm':	result /= 10; /* fall through */
		case 'c':	result /= 10; /* fall through */
		case 'd':	result /= 10; break;
	}

	return result;
}

void print_help() {
	printf("Usage: %s -b beginning -e end -i increment [-c column] [-q] file [more files]\n", H_PROGNAME);
	printf("\n");
	printf("Options:\n");
	printf(" -b	Start counting at this number.\n");
	printf(" -e	Stop counting at this number.\n");
	printf(" -i	Count by this amount.\n");
	printf(" -q	Quiet output.\n");
	printf(" -c	Read data from this column of the input files.\n");
}

int quiet = FALSE;

int main(int argc, char *argv[]) {
	int c;
	double d, begin, end, incr, distance;
	int slot_count, column = 1, filecount, result, j;
	histogram_t *histograms;

	begin = end = incr = 0.0;

	while(1) {
		c = getopt(argc, argv, "c:b:e:i:q");

		if(c == -1) {
			break;
		}

		switch(c) {
			case 'c':	//column number
				column = getNumber(optarg);
				break;
			case 'b':	//beginning
				begin = getNumber(optarg);
				break;
			case 'e':	//end
				end = getNumber(optarg);
				break;
			case 'i':	//increment
				incr = getNumber(optarg);
				break;
			case 'q':
				quiet = TRUE;
				break;
			default:
				print_help();
				return 1;
		}
	}

	if( (begin*end == 0) && (begin - end == 0) ) {
		fprintf(stderr, "%s: begin and end both cannot be zero.\n", H_PROGNAME);
		print_help();
		return 1;
	}

	if(end < begin) {
		d = begin;
		begin = end;
		end = d;
	}

	if( incr <= 0 ) {
		fprintf(stderr, "%s: increment must be positive.\n", H_PROGNAME);
		print_help();
		return 1;
	}

	distance = fabs(end - begin);

	if( incr > distance ) {
		fprintf(stderr, "%s: increment must be smaller than abs(end - begin).\n", H_PROGNAME);
		print_help();
		return 1;
	}

	filecount = argc - optind;
	
	if( filecount < 1 ) {
		fprintf(stderr, "%s: Please provide some file names.\n", H_PROGNAME);
		print_help();
		return 1;
	}

	//figure out how many slots we need
	slot_count = ceil(distance / incr);

	//printf("%d %f %f %f\n", slot_count, begin, end, incr);

	histograms = allocateHistograms(filecount, slot_count);

	//do our magic counting here.
	for(c = 0, result = 0; c < filecount; c++) {
		result += countSamples(argv[c + optind], &histograms[c], begin, end, incr, column);
	}

	//now print results.
	if(result > 0) {
		//print out leftmost column...
		printf("#Samples");

		for(j = 0; j < filecount; j++) {
			if(histograms[j].label != NULL) {
				printf(", %s", histograms[j].label);
			}
		}
		
		printf("\n");

		//now go through each row
		for(c = 0, d = begin; c < slot_count; c++, d += incr) {
			printf("%.2f", d + (incr / 2));

			//go print out each cell
			for(j = 0; j < filecount; j++) {
				if(histograms[j].label != NULL) {
					printf(", %.2f", (double)100 * histograms[j].samples[c] / histograms[j].samples_taken);
				}
			}

			printf("\n");
		}
	}

#if 0
	freeHistograms(&histograms, filecount, slot_count);
#endif

	return 0;
}

histogram_t *allocateHistograms(int count, int samples) {
	int i;
	histogram_t *d = malloc( sizeof(histogram_t) * count);

	for(i = 0; i < count; i++) {
		d[i].label = NULL;
		d[i].samples = malloc( sizeof(int) * samples );
		memset(d[i].samples, 0, sizeof(int) * samples );
		d[i].samples_taken = 0;
	}

	return d;
}

#if 0
void freeHistograms(histogram_t **ptr, int count, int samples) {
	int i;
	histogram_t *d = *ptr;

	for(i = 0; i < count; i++) {
		if(d[i].label != NULL) {
			free(d[i].label);
		}
		free(d[i].samples);
	}
	free(d);

	*ptr = NULL;
}
#endif

char newstr[8192];

void trim(char *str) {
	int sz, off, end;
	char *newstr;

#ifdef DEBUG
	fprintf(stderr, "trim '%s' %d\n", str, strlen(str));
#endif

	if(str == NULL) {
		return;
	}

	sz = strlen(str);

	for(off = 0; off < sz && isspace(str[off]) != 0; off++) {
	}

	for(end = sz - 1; end >= 0 && isspace(str[end]) != 0; end--) {
	}

	if(end < 0) {	//if we get here, the string's empty.
		str[0] = 0;
		return;
	}

	sz = end - off + 10;
#ifdef DEBUG
	fprintf(stderr, "%d %d %d\n", off, end, sz);
#endif

	newstr = malloc(sz);
	strncpy(newstr, str + off, end - off + 1);
	newstr[end - off + 1] = 0;
	strcpy(str, newstr);
	free(newstr);

#ifdef DEBUG
	fprintf(stderr, "got to end of trim\n");
#endif
}

int countSamples(const char *fname, histogram_t *record, double begin, double end, double incr, int column) {
	//char *lbl;
	FILE *file;
	char *token, *tmpvar;
	int toknum, slot_count;
	double dummy, distance;
	char line[LINESIZE];
	int slot, samples_taken;

	//open file
	file = fopen(fname, "r");
	if(file == NULL) {
		record->label = NULL;
		fprintf(stderr, "Data file %s: %s.\n", fname, strerror(errno));
		return 0;
	}

	//first copy file name in as label.
	record->label = malloc(strlen(fname) + 2);
	strcpy(record->label, fname);

	distance = end - begin;
	slot_count = ceil(distance / incr);

	samples_taken = 0;

	//read data
	while(fgets(line, LINESIZE, file) != NULL) {
#ifdef DEBUG
		fprintf(stderr, "read: %p ", line);
		fprintf(stderr, " '%s'\n", line);
#endif
		trim(line);
		if(line[0] == '#') continue;

		tmpvar = line;

		toknum = 0;
		while( (token = strtok(tmpvar, " ,")) != NULL) {
			toknum++;
			tmpvar = NULL;

			//if this isn't the one we want, go on
			if(toknum != column) continue;

			dummy = getNumber(token);

			//now go figure out where to put it
			if(FRAME(begin, end, dummy)) {
				//figure out which bin to place the value in.
				dummy -= begin;
				slot = (dummy * (slot_count / distance));

				record->samples[slot] += 1;

				samples_taken++;
			} else if(quiet == FALSE) {
				fprintf(stderr, "%s: Value %f is out of range!\n", fname, dummy);
			}
		}
	}

	record->samples_taken = samples_taken;

	fclose(file);
	return samples_taken;
}
